<?php
/**
 * Download Handler - Multi-Bot File Download System
 * Version: 3.5
 * Author: DeepSeek
 */

require_once __DIR__ . '/../core/FileManager.php';
require_once __DIR__ . '/../core/Database.php';
require_once __DIR__ . '/../core/Bot.php';

class DownloadHandler {
    private $bot;
    private $fileManager;
    private $db;
    private $update;
    private $message;
    private $chatId;
    private $userId;
    private $requestId;

    public function __construct(Bot $bot, array $update) {
        $this->bot = $bot;
        $this->fileManager = new FileManager();
        $this->db = Database::getInstance();
        $this->update = $update;
        $this->message = $update['message'] ?? $update['channel_post'] ?? null;
        
        if ($this->message) {
            $this->chatId = $this->message['chat']['id'];
            $this->userId = $this->message['from']['id'] ?? 0;
        }
    }

    /**
     * هندل کردن درخواست‌های دانلود
     */
    public function handle() {
        if (!$this->message) return false;

        $text = $this->message['text'] ?? '';
        
        // استخراج شناسه فایل از دستور /download یا مستقیماً از لینک
        $this->requestId = $this->extractFileId($text);
        
        if (!$this->requestId) {
            $this->sendUsageInstructions();
            return false;
        }

        $this->bot->sendChatAction('upload_document');
        return $this->processDownload();
    }

    /**
     * استخراج شناسه فایل از متن پیام
     */
    private function extractFileId($text) {
        // حالت اول: دستور /download به همراه لینک
        if (preg_match('/\/download\s+(https?:\/\/\S+)/i', $text, $matches)) {
            return $this->fileManager->extractFileId($matches[1]);
        }
        
        // حالت دوم: فقط لینک ارسال شده
        if (filter_var($text, FILTER_VALIDATE_URL)) {
            return $this->fileManager->extractFileId($text);
        }
        
        // حالت سوم: شناسه مستقیم
        if (preg_match('/^[a-f0-9]{32}$/i', $text)) {
            return $text;
        }
        
        return null;
    }

    /**
     * پردازش عملیات دانلود
     */
    private function processDownload() {
        try {
            // اعتبارسنجی شناسه فایل
            $validation = $this->fileManager->validateFileId($this->requestId);
            if ($validation !== true) {
                throw new Exception($validation);
            }

            // دریافت اطلاعات فایل
            $fileInfo = $this->fileManager->getFileInfo($this->requestId);
            if (!$fileInfo) {
                throw new Exception("File not found");
            }

            // ارسال فایل به کاربر
            $this->bot->sendDocument(
                $this->chatId,
                $fileInfo['file_path'],
                $fileInfo['file_name']
            );

            // ثبت لاگ دانلود
            $this->logDownload($fileInfo['id']);

            // افزایش شمارنده دانلود
            $this->fileManager->incrementDownloadCount($fileInfo['id']);

            return true;
        } catch (Exception $e) {
            $this->handleDownloadError($e->getMessage());
            return false;
        }
    }

    /**
     * ثبت اطلاعات دانلود در دیتابیس
     */
    private function logDownload($fileId) {
        $this->db->logDownload(
            $fileId,
            $this->bot->getBotId(),
            $this->userId
        );
    }

    /**
     * ارسال راهنمای استفاده
     */
    private function sendUsageInstructions() {
        $message = "📥 برای دانلود فایل، یکی از روش‌های زیر را استفاده کنید:\n\n";
        $message .= "1. لینک دانلود را مستقیم ارسال کنید\n";
        $message .= "2. از دستور زیر استفاده کنید:\n";
        $message .= "<code>/download لینک_دانلود</code>\n\n";
        $message .= "مثال:\n<code>/download https://example.com/download?id=file_123abc</code>";

        $this->bot->sendMessage(
            $this->chatId,
            $message,
            'HTML'
        );
    }

    /**
     * هندل خطاهای دانلود
     */
    private function handleDownloadError($error) {
        $errorMessages = [
            'File not found' => "❌ فایل مورد نظر یافت نشد",
            'Physical file missing' => "❌ فایل در سرور وجود ندارد",
            'File has expired' => "❌ لینک دانلود منقضی شده است",
            'Invalid file ID' => "❌ شناسه فایل نامعتبر است"
        ];

        $userMessage = $errorMessages[$error] ?? "❌ خطا در دریافت فایل: $error";
        
        $this->bot->sendMessage(
            $this->chatId,
            $userMessage,
            null,
            true
        );
        
        // اگر فایل منقضی شده، اطلاعات بیشتری بده
        if ($error === 'File has expired') {
            $this->bot->sendMessage(
                $this->chatId,
                "⚠️ لینک‌های دانلود به مدت ۷۲ ساعت معتبر هستند. برای دریافت لینک جدید فایل را مجدداً آپلود کنید.",
                null,
                true
            );
        }
    }
}
?>