<?php
/**
 * Main Entry Point - Uploader Pro Multi-Bot System
 * Version: 4.0
 * Author: DeepSeek
 * Description: Core router and multi-bot manager
 */

// =====================================
//  تنظیمات اولیه و امنیتی
// =====================================
declare(strict_types=1);
session_write_close(); // بهبود عملکرد در ترافیک بالا

// غیرفعال کردن گزارش خطا در حالت عملیاتی
error_reporting(0);
ini_set('display_errors', '0');

// بارگذاری تنظیمات اصلی
require_once __DIR__ . '/config.php';

// =====================================
//  خودکار بارگذاری کلاس‌ها
// =====================================
spl_autoload_register(function ($class) {
    $paths = [
        __DIR__ . '/core/',
        __DIR__ . '/handlers/'
    ];
    
    foreach ($paths as $path) {
        $file = $path . $class . '.php';
        if (file_exists($file)) {
            require_once $file;
            return;
        }
    }
    
    // برای محیط توسعه
    if (defined('DEV_MODE') && DEV_MODE) {
        error_log("Class $class not found");
    }
});

// =====================================
//  مدیریت خطاها
// =====================================
set_exception_handler(function (Throwable $e) {
    error_log("Uncaught Exception: " . $e->getMessage() . " in " . $e->getFile() . ":" . $e->getLine());
    
    if (defined('DEV_MODE') && DEV_MODE) {
        header('Content-Type: application/json');
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage(),
            'trace' => $e->getTrace()
        ]);
    } else {
        http_response_code(500);
        echo "Internal Server Error";
    }
    
    exit;
});

// =====================================
//  تابع‌های کمکی
// =====================================

/**
 * بررسی اعتبار درخواست
 */
function validateRequest(): void {
    // جلوگیری از حملات CSRF (اختیاری)
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($_SERVER['HTTP_REFERER'])) {
        throw new Exception("Invalid request source");
    }
    
    // محدودیت درخواست‌ها در دقیقه
    static $requestCount = 0;
    static $lastReset = 0;
    
    $currentTime = time();
    if ($currentTime - $lastReset > 60) {
        $requestCount = 0;
        $lastReset = $currentTime;
    }
    
    if (++$requestCount > MAX_REQUESTS_PER_MINUTE) {
        http_response_code(429);
        exit("Too many requests");
    }
}

// =====================================
//  بدنه اصلی برنامه
// =====================================
try {
    // اعتبارسنجی اولیه
    validateRequest();
    
    // دریافت آپدیت از تلگرام
    $input = file_get_contents('php://input');
    $update = json_decode($input, true);
    
    if (!$update) {
        throw new Exception("Invalid Telegram update");
    }
    
    // =====================================
    //  سیستم مدیریت چندرباتی (Multi-Bot)
    // =====================================
    $db = Database::getInstance();
    
    // 1. دریافت ربات‌های فعال از دیتابیس
    $activeBots = $db->getActiveBots();
    
    // 2. اگر ربات فعالی نبود، از تنظیمات استفاده کن
    if (empty($activeBots)) {
        global $botTokens;
        $activeBots = [];
        
        foreach ($botTokens as $token) {
            $activeBots[] = [
                'token' => $token,
                'id' => md5($token) // ID موقت
            ];
        }
    }
    
    // 3. ارسال آپدیت به ربات‌ها به ترتیب
    $handled = false;
    $lastError = null;
    
    foreach ($activeBots as $botInfo) {
        try {
            $bot = new Bot($botInfo['token'], $botInfo['id']);
            $bot->setUpdate($update);
            
            if ($bot->start()) {
                $handled = true;
                break; // خروج بعد از اولین ربات موفق
            }
        } catch (Exception $e) {
            $lastError = $e->getMessage();
            error_log("Bot {$botInfo['id']} failed: " . $e->getMessage());
            
            // غیرفعال کردن ربات مشکل‌دار
            $db->setBotStatus($botInfo['token'], 'inactive');
            continue;
        }
    }
    
    // 4. اگر هیچ رباتی پاسخ نداد
    if (!$handled) {
        error_log("All bots failed to handle update. Last error: $lastError");
        http_response_code(503);
        exit("Service Unavailable");
    }
    
    // پاسخ موفق به تلگرام
    http_response_code(200);
    echo "OK";
    
} catch (Throwable $e) {
    // مدیریت خطاهای کلی
    http_response_code(500);
    error_log("Fatal Error: " . $e->getMessage());
    echo "Internal Server Error";
}