<?php
/**
 * Upload Handler - Multi-Bot File Upload System
 * Version: 3.0
 * Author: DeepSeek
 */

require_once __DIR__ . '/../core/FileManager.php';
require_once __DIR__ . '/../core/Database.php';
require_once __DIR__ . '/../core/Bot.php';

class UploadHandler {
    private $bot;
    private $fileManager;
    private $db;
    private $update;
    private $message;
    private $chatId;
    private $userId;

    public function __construct(Bot $bot, array $update) {
        $this->bot = $bot;
        $this->fileManager = new FileManager();
        $this->db = Database::getInstance();
        $this->update = $update;
        $this->message = $update['message'] ?? $update['channel_post'] ?? null;
        
        if ($this->message) {
            $this->chatId = $this->message['chat']['id'];
            $this->userId = $this->message['from']['id'] ?? 0;
        }
    }

    /**
     * تشخیص نوع فایل و هندل آپلود
     */
    public function handle() {
        if (!$this->message) return false;

        $document = $this->message['document'] ?? null;
        $photo = $this->message['photo'] ?? null;
        $video = $this->message['video'] ?? null;
        $audio = $this->message['audio'] ?? null;
        $voice = $this->message['voice'] ?? null;

        if ($document) {
            return $this->handleDocument($document);
        } elseif ($photo) {
            return $this->handlePhoto(end($photo)); // گرفتن بالاترین کیفیت
        } elseif ($video) {
            return $this->handleVideo($video);
        } elseif ($audio) {
            return $this->handleAudio($audio);
        } elseif ($voice) {
            return $this->handleVoice($voice);
        }

        return false;
    }

    /**
     * هندل آپلود اسناد (PDF, Word, Zip, etc.)
     */
    private function handleDocument($document) {
        $fileId = $document['file_id'];
        $fileName = $document['file_name'] ?? 'document_' . time();
        $fileSize = $document['file_size'] ?? 0;

        $this->bot->sendChatAction('typing');

        try {
            // دریافت اطلاعات فایل از تلگرام
            $fileInfo = $this->bot->apiRequest('getFile', ['file_id' => $fileId]);
            if (!$fileInfo || !$fileInfo['ok']) {
                throw new Exception("Failed to get file info from Telegram");
            }

            $filePath = $fileInfo['result']['file_path'];
            $telegramFileUrl = "https://api.telegram.org/file/bot{$this->bot->getToken()}/{$filePath}";

            // ذخیره فایل
            $savedFileId = $this->fileManager->saveFile(
                $telegramFileUrl,
                $fileName,
                $this->userId
            );

            // ارسال لینک دانلود
            $downloadLink = $this->fileManager->getDownloadLink($savedFileId);
            $this->sendSuccessResponse($fileName, $downloadLink, $fileSize);

            return $savedFileId;
        } catch (Exception $e) {
            $this->handleUploadError($e->getMessage());
            return false;
        }
    }

    /**
     * هندل آپلود عکس
     */
    private function handlePhoto($photo) {
        $fileId = $photo['file_id'];
        $fileSize = $photo['file_size'] ?? 0;
        $fileName = 'photo_' . time() . '.jpg';

        $this->bot->sendChatAction('upload_photo');

        try {
            // دریافت اطلاعات فایل از تلگرام
            $fileInfo = $this->bot->apiRequest('getFile', ['file_id' => $fileId]);
            if (!$fileInfo || !$fileInfo['ok']) {
                throw new Exception("Failed to get file info from Telegram");
            }

            $filePath = $fileInfo['result']['file_path'];
            $telegramFileUrl = "https://api.telegram.org/file/bot{$this->bot->getToken()}/{$filePath}";

            // ذخیره فایل
            $savedFileId = $this->fileManager->saveFile(
                $telegramFileUrl,
                $fileName,
                $this->userId
            );

            // ارسال لینک دانلود
            $downloadLink = $this->fileManager->getDownloadLink($savedFileId);
            $this->sendSuccessResponse($fileName, $downloadLink, $fileSize);

            return $savedFileId;
        } catch (Exception $e) {
            $this->handleUploadError($e->getMessage());
            return false;
        }
    }

    /**
     * هندل آپلود ویدیو
     */
    private function handleVideo($video) {
        $fileId = $video['file_id'];
        $fileName = $video['file_name'] ?? 'video_' . time() . '.mp4';
        $fileSize = $video['file_size'] ?? 0;

        $this->bot->sendChatAction('upload_video');

        try {
            // دریافت اطلاعات فایل از تلگرام
            $fileInfo = $this->bot->apiRequest('getFile', ['file_id' => $fileId]);
            if (!$fileInfo || !$fileInfo['ok']) {
                throw new Exception("Failed to get file info from Telegram");
            }

            $filePath = $fileInfo['result']['file_path'];
            $telegramFileUrl = "https://api.telegram.org/file/bot{$this->bot->getToken()}/{$filePath}";

            // ذخیره فایل
            $savedFileId = $this->fileManager->saveFile(
                $telegramFileUrl,
                $fileName,
                $this->userId
            );

            // ارسال لینک دانلود
            $downloadLink = $this->fileManager->getDownloadLink($savedFileId);
            $this->sendSuccessResponse($fileName, $downloadLink, $fileSize);

            return $savedFileId;
        } catch (Exception $e) {
            $this->handleUploadError($e->getMessage());
            return false;
        }
    }

    /**
     * هندل آپلود فایل‌های صوتی
     */
    private function handleAudio($audio) {
        $fileId = $audio['file_id'];
        $fileName = $audio['file_name'] ?? 'audio_' . time() . '.mp3';
        $fileSize = $audio['file_size'] ?? 0;

        $this->bot->sendChatAction('upload_audio');

        try {
            // دریافت اطلاعات فایل از تلگرام
            $fileInfo = $this->bot->apiRequest('getFile', ['file_id' => $fileId]);
            if (!$fileInfo || !$fileInfo['ok']) {
                throw new Exception("Failed to get file info from Telegram");
            }

            $filePath = $fileInfo['result']['file_path'];
            $telegramFileUrl = "https://api.telegram.org/file/bot{$this->bot->getToken()}/{$filePath}";

            // ذخیره فایل
            $savedFileId = $this->fileManager->saveFile(
                $telegramFileUrl,
                $fileName,
                $this->userId
            );

            // ارسال لینک دانلود
            $downloadLink = $this->fileManager->getDownloadLink($savedFileId);
            $this->sendSuccessResponse($fileName, $downloadLink, $fileSize);

            return $savedFileId;
        } catch (Exception $e) {
            $this->handleUploadError($e->getMessage());
            return false;
        }
    }

    /**
     * هندل آپلود پیام‌های صوتی
     */
    private function handleVoice($voice) {
        $fileId = $voice['file_id'];
        $fileSize = $voice['file_size'] ?? 0;
        $fileName = 'voice_' . time() . '.ogg';

        $this->bot->sendChatAction('upload_audio');

        try {
            // دریافت اطلاعات فایل از تلگرام
            $fileInfo = $this->bot->apiRequest('getFile', ['file_id' => $fileId]);
            if (!$fileInfo || !$fileInfo['ok']) {
                throw new Exception("Failed to get file info from Telegram");
            }

            $filePath = $fileInfo['result']['file_path'];
            $telegramFileUrl = "https://api.telegram.org/file/bot{$this->bot->getToken()}/{$filePath}";

            // ذخیره فایل
            $savedFileId = $this->fileManager->saveFile(
                $telegramFileUrl,
                $fileName,
                $this->userId
            );

            // ارسال لینک دانلود
            $downloadLink = $this->fileManager->getDownloadLink($savedFileId);
            $this->sendSuccessResponse($fileName, $downloadLink, $fileSize);

            return $savedFileId;
        } catch (Exception $e) {
            $this->handleUploadError($e->getMessage());
            return false;
        }
    }

    /**
     * ارسال پاسخ موفقیت‌آمیز به کاربر
     */
    private function sendSuccessResponse($fileName, $downloadLink, $fileSize) {
        $sizeMB = round($fileSize / 1024 / 1024, 2);
        
        $message = "✅ فایل شما با موفقیت آپلود شد!\n\n";
        $message .= "📄 نام فایل: `" . htmlspecialchars($fileName) . "`\n";
        $message .= "📦 حجم فایل: " . ($sizeMB > 0 ? "$sizeMB مگابایت" : "نامشخص") . "\n\n";
        $message .= "🔗 لینک دائمی دانلود:\n";
        $message .= "<code>$downloadLink</code>\n\n";
        $message .= "این لینک را میتوانید با دیگران به اشتراک بگذارید و از هر رباتی قابل دانلود است!";

        $this->bot->sendMessage(
            $this->chatId,
            $message,
            'HTML',
            true,
            null,
            [
                [
                    ['text' => "📥 دانلود فایل", 'url' => $downloadLink],
                    ['text' => "🤖 ربات اصلی", 'url' => "https://t.me/{$this->bot->getBotUsername()}"]
                ]
            ]
        );
    }

    /**
     * هندل خطاهای آپلود
     */
    private function handleUploadError($error) {
        $errorMessages = [
            'File size exceeds limit' => "❌ حجم فایل بیشتر از حد مجاز است (حداکثر 20 مگابایت)",
            'File type not allowed' => "❌ نوع فایل مجاز نیست",
            'Failed to save file to server' => "❌ خطا در ذخیره فایل. لطفاً دوباره تلاش کنید",
            'Database error' => "❌ خطای پایگاه داده. لطفاً به پشتیبانی اطلاع دهید"
        ];

        $userMessage = $errorMessages[$error] ?? "❌ خطا در آپلود فایل: $error";
        
        $this->bot->sendMessage(
            $this->chatId,
            $userMessage,
            null,
            true
        );
    }
}
?>