<?php
/**
 * Download Gateway - Secure File Delivery System
 * Version: 4.0
 * Author: DeepSeek
 * Security Level: High
 */

// =====================================
//  تنظیمات امنیتی و اولیه
// =====================================
declare(strict_types=1);
session_write_close();
ignore_user_abort(false);

// غیرفعال کردن گزارش خطا در حالت عملیاتی
error_reporting(0);
ini_set('display_errors', '0');

// بارگذاری تنظیمات و کلاس‌های ضروری
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/core/Database.php';
require_once __DIR__ . '/core/FileManager.php';

// =====================================
//  مدیریت خطاها
// =====================================
set_exception_handler(function (Throwable $e) {
    error_log("Download Error: " . $e->getMessage() . " in " . $e->getFile() . ":" . $e->getLine());
    
    if (defined('DEV_MODE') && DEV_MODE) {
        header('Content-Type: text/plain');
        echo "DEBUG MODE:\n" . $e->getMessage();
    } else {
        header('HTTP/1.1 500 Internal Server Error');
        readfile(__DIR__ . '/templates/error.html');
    }
    
    exit;
});

// =====================================
//  توابع امنیتی و کمکی
// =====================================

/**
 * ضدعفونی کردن ورودی‌ها
 */
function sanitizeInput(string $input): string {
    return htmlspecialchars(strip_tags($input), ENT_QUOTES, 'UTF-8');
}

/**
 * محدودیت درخواست‌ها برای جلوگیری از حملات Brute Force
 */
function applyRateLimit(): void {
    $clientIP = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $cacheFile = __DIR__ . "/cache/rate_limit_$clientIP.txt";
    
    $currentTime = time();
    $lastRequest = file_exists($cacheFile) ? (int)file_get_contents($cacheFile) : 0;
    
    if ($currentTime - $lastRequest < ANTI_FLOOD_DELAY) {
        header('HTTP/1.1 429 Too Many Requests');
        header('Retry-After: ' . ANTI_FLOOD_DELAY);
        exit("Too many requests. Please wait and try again.");
    }
    
    file_put_contents($cacheFile, $currentTime);
}

// =====================================
//  بدنه اصلی دانلود
// =====================================
try {
    // اعمال محدودیت نرخ درخواست
    applyRateLimit();
    
    // دریافت شناسه فایل
    $fileId = $_GET['id'] ?? '';
    if (empty($fileId)) {
        header('HTTP/1.1 400 Bad Request');
        exit("Missing file ID");
    }
    
    // ضدعفونی شناسه فایل
    $fileId = sanitizeInput($fileId);
    
    // اعتبارسنجی شناسه فایل
    if (!preg_match('/^[a-f0-9]{32}$/i', $fileId)) {
        header('HTTP/1.1 400 Bad Request');
        exit("Invalid file ID format");
    }
    
    // ایجاد نمونه‌های لازم
    $fileManager = new FileManager();
    $db = Database::getInstance();
    
    // اعتبارسنجی کامل فایل
    $validation = $fileManager->validateFileId($fileId);
    if ($validation !== true) {
        header('HTTP/1.1 404 Not Found');
        exit($validation);
    }
    
    // دریافت اطلاعات فایل
    $fileInfo = $fileManager->getFileInfo($fileId);
    if (!$fileInfo) {
        header('HTTP/1.1 404 Not Found');
        exit("File not found in database");
    }
    
    $filePath = $fileInfo['file_path'];
    $fileName = $fileInfo['file_name'];
    
    // بررسی وجود فایل فیزیکی
    if (!file_exists($filePath) {
        header('HTTP/1.1 404 Not Found');
        exit("Physical file missing");
    }
    
    // افزایش شمارنده دانلود
    $fileManager->incrementDownloadCount($fileId);
    
    // =====================================
    //  ارسال فایل به کاربر
    // =====================================
    $fileSize = filesize($filePath);
    $fileMime = mime_content_type($filePath);
    
    // هدرهای لازم برای دانلود امن
    header('HTTP/1.1 200 OK');
    header('Content-Type: ' . $fileMime);
    header('Content-Disposition: attachment; filename="' . rawurlencode($fileName) . '"');
    header('Content-Length: ' . $fileSize);
    header('Content-Transfer-Encoding: binary');
    header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
    header('Pragma: public');
    header('Expires: 0');
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: DENY');
    header('X-XSS-Protection: 1; mode=block');
    
    // بهینه‌سازی ارسال فایل‌های حجیم
    if ($fileSize > 1024 * 1024) { // >1MB
        $chunkSize = 1024 * 1024; // 1MB chunks
        
        $handle = fopen($filePath, 'rb');
        if (!$handle) throw new Exception("Failed to open file");
        
        while (!feof($handle)) {
            echo fread($handle, $chunkSize);
            ob_flush();
            flush();
            
            // بررسی اتصال کاربر
            if (connection_aborted()) break;
        }
        
        fclose($handle);
    } else {
        // ارسال مستقیم برای فایل‌های کوچک
        readfile($filePath);
    }
    
    // ثبت لاگ دانلود موفق
    $referer = $_SERVER['HTTP_REFERER'] ?? 'direct';
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    
    $logMessage = sprintf(
        "[%s] DOWNLOAD: %s | IP: %s | Referer: %s | UA: %s\n",
        date('Y-m-d H:i:s'),
        $fileId,
        $ipAddress,
        $referer,
        $userAgent
    );
    
    file_put_contents(__DIR__ . '/logs/downloads.log', $logMessage, FILE_APPEND);
    
    exit;

} catch (Throwable $e) {
    // مدیریت خطاهای کلی
    header('HTTP/1.1 500 Internal Server Error');
    error_log("Download Gateway Error: " . $e->getMessage());
    readfile(__DIR__ . '/templates/error.html');
    exit;
}